/*
  r2sync -- synchronize two files or directories, one of which may be remote

  Usage: r2sync [options] target1 target2
     or: r2sync [options] profile
     or: r2sync [-p port] -d

  r2sync is called with as arguments two files or directories, which
  may be remote. It forks two other r2sync processes in daemon (-d)
  mode, using SSH to start r2sync on remote machines, or connecting to
  an already running r2sync server. Those processes are asked to check
  which of the files or directories have been updated since the last
  synchronization. It then asks the user (unless it's running in batch
  mode) for each file or directory if it needs to be synchronized and
  if so, in which direction. It then uses the rsync algorithm to
  synchronize the files as efficiently as possible. (The algorithm
  tries to avoid copying parts of a file that are the same on both
  sides.)

  Each target can be:

    - a local path (relative or absolute),

    - a path (relative or absolute) on a remote host: "host:path" or
      "user@host:path", where relative paths are relative to the home
      directory on that host, or

    - an r2sync URL "r2sync://host/path" or "r2sync://host:port/path"
      to connect to an already running r2sync server (default port
      874) with an absolute path.

  It is also possible to create a named profile, which is a file that
  contains the names of the two targets. That is useful for
  synchronizations that are done often, especially as the profile can
  also store the name patterns of files that should be skipped.

  This program is inspired by and aims to be a viable replacement for
  unison, but, unlike unison, it has the extra aims that r2sync's
  network protocol is well-documented and independent of the r2sync
  version (or the compiler used to compile it); and that future
  versions or r2sync remain backwards compatible.

  Copyright © 2018 Bert Bos <bert@phonk.net>
  Created: 28 January 2018
  Author: Bert Bos <bert@phonk.net>
*/

#include "stdincls.h"
#include "types.e"
#include "print.e"
#include "c-profile.e"
#include "c-client.e"
#include "s-serve.e"

static struct option options[] = {
  {"autocheck", no_argument, NULL, 'a'},
  {"batch", no_argument, NULL, 'b'},
  {"compression", no_argument, NULL, 'C'},
  {"conflict-resolution", required_argument, NULL, 'c'},
  {"daemon", no_argument, NULL, 'd'},
  {"exclude", required_argument, NULL, 'e'},
  {"help", no_argument, NULL, 'h'},
  {"include", required_argument, NULL, 'i'},
  {"keep-going", no_argument, NULL, 'k'},
  {"noshortcuts", no_argument, NULL, 'n'},
  {"port", required_argument, NULL, 'p'},
  {"protocol", required_argument, NULL, 'P'},
  {"quiet", no_argument, NULL, 'q'},
  {"reset-logs", no_argument, NULL, 'r'},
  {"statistics", no_argument, NULL, 's'},
  {"version", no_argument, NULL, 'v'},
  {NULL, 0, NULL, 0}
};


/* usage -- print usage message and exit */
static void usage(bool on_stderr)
{
  print1(on_stderr ? stderr : stdout,
	_("Usage: r2sync [options] target1 target2\n\
   or: r2sync [options] profile-name\n\
   or: r2sync [-p port] -d\n"));
  exit(on_stderr ? EX_USAGE : 0);
}


/* handle_signal -- set a flag and pass signals on to our children */
static void handle_signal(int number)
{
  /* SIGALRM is used by get_protocol(). Ignore it (except that it will
     interrupt a read() call). Otherwise, if we already handled a
     signal, ignore this one. */
  if (number != SIGALRM && signaled == 0) {
    signaled = number;	    /* Set a flag */
    kill(0, number);	    /* Broadcast to group (self & children) */
  }
}


/* main -- main body */
int main(int argc, char *argv[])
{
  char *port = NULL, buf1[PATH_MAX+1], buf2[PATH_MAX+1];
  struct profile profile;
  bool as_daemon = false;
  struct sigaction sa;
  int c, longindex;

  /* For sorting, printing thousands separators, counting chars in status() */
  (void) setlocale(LC_ALL, "");
  (void) bindtextdomain(PACKAGE, LOCALEDIR);
  (void) textdomain(PACKAGE);

  /* Check syntax of command line options and check for -d (--daemon) */
  while ((c = getopt_long(argc, argv, "CP:abc:de:hi:knp:qrsv",
			  options, &longindex)) != -1)
    switch (c) {
    case 'd': as_daemon = true; break;
    case 'h': usage(false);
    case 'v': printf(_("%1$s (protocol %2$d)\n"), PACKAGE_STRING, R2SYNCPROTO);
      return 0;
    case '?': return EX_USAGE;
    }

  /* Set some signal handlers */
  sigemptyset(&sa.sa_mask);
  sa.sa_flags = 0;		/* Do not set SA_RESTART */
  sa.sa_handler = handle_signal;
  (void) sigaction(SIGINT, &sa, NULL);
  (void) sigaction(SIGHUP, &sa, NULL);
  (void) sigaction(SIGTERM, &sa, NULL);
  (void) sigaction(SIGPIPE, &sa, NULL);
  (void) sigaction(SIGALRM, &sa, NULL);

  if (as_daemon) {		/* Daemon mode */

    /* Check for corresponding options and then run in daemon mode */
    optind = 1;
    while ((c = getopt_long(argc, argv, ":dp:", options, &longindex)) != -1)
      switch (c) {
      case 'd': break;		/* Already handled */
      case 'p': port = optarg; break;
      default:
	if (optopt)
	  errx(EX_USAGE, _("Option '-%c' not allowed in daemon mode (-d)"),
	       optopt);
	else
	  errx(EX_USAGE, _("Option '--%s' not allowed in daemon mode (-d)"),
	       options[longindex].name);
      }

    return !serve(0, 1, port);

  } else {			/* Master mode */

    /* Create a process group so kill(0,...) affects this process and
       its children, but not the parent (probably a shell). */
    if (setpgid(0, getpid()) != 0) err(1, NULL);

    /* Initialize the fields of the profile */
    init_profile(&profile);

    /* Check the number of arguments, set the roots or the profile name. */
    if (argc == optind + 1) {
      if (strchr(argv[optind], '/'))
	errx(EX_USAGE, _("The profile name must be a simple name, not a path"));
      profile.profile = argv[optind];
      read_profile(&profile);
    } else if (argc == optind + 2) {
      if (argv[optind][strcspn(argv[optind], ":/")] == ':')
	profile.root1 = strdup(argv[optind]); /* Remote target */
      else if (realpath(argv[optind], buf1))
	profile.root1 = strdup(buf1);
      else
	err(EX_OSERR, "'%s'", argv[optind]);
      if (argv[optind+1][strcspn(argv[optind+1], ":/")] == ':')
	profile.root2 = strdup(argv[optind+1]); /* Remote target */
      else if (realpath(argv[optind+1], buf2))
	profile.root2 = strdup(buf2);
      else
	err(EX_OSERR, "'%s'", argv[optind+1]);
    } else
      usage(true);

    /* Sanity check */
    if (strcmp(profile.root1, profile.root2) == 0)
      errx(EX_USAGE, _("Cannot sync a target with itself"));

    /* Now parse command line options again, overriding the profile */
    optind = 1;
    while ((c = getopt_long(argc, argv, ":CP:abc:e:i:knqrs",
			    options, &longindex)) != -1)
      switch (c) {
      case 'C': profile.compression = true; break;
      case 'P': profile.protocol = atoi(optarg); break;
      case 'a': profile.autocheck = true; break;
      case 'b': profile.batch_mode = true; break;
      case 'c': profile.conflict_resolution = *optarg; break;
      case 'e': add_exclusion(&profile, optarg); break;
      case 'i': add_inclusion(&profile, optarg); break;
      case 'k': profile.keep_going = true; break;
      case 'n': profile.use_shortcuts = false; break;
      case 'q': profile.quiet = true; break;
      case 'r': profile.reset_logs = true; break;
      case 's': profile.statistics = true; break;
      default:
	if (optopt)
	  errx(EX_USAGE, _("Option '-%c' only valid in daemon mode (-d)"),
	       optopt);
	else
	  errx(EX_USAGE, _("Option '--%s' only valid in daemon mode (-d)"),
	       options[longindex].name);
      }

    if (profile.protocol < 1)
      errx(EX_USAGE, _("The protocol (-P) must be a positive number"));
    if (profile.protocol > R2SYNCPROTO)
      errx(EX_USAGE, _("The protocol (-P) cannot exceed %d"), R2SYNCPROTO);
    if (!strchr("/<>", profile.conflict_resolution))
      errx(EX_USAGE,_("The conflict resolution action (-c) must be <, > or /"));
    if (profile.autocheck && !profile.use_shortcuts)
      errx(EX_USAGE, _("Autocheck (-a) and noshortcuts (-n) are incompatible"));

    /* We could free() the roots and exclusions, but we're exiting anyway... */
    return !client(&profile);
  }
}
