/*
  determine_file_actions() is called by client() to determine what to
  do with each pair of files that is unequal. It asks the user (unless
  r2sync is running in in batch mode) and returns lists of files that
  need to be updated on each of the servers and a list of files that
  need to be deleted.

  Implementation:

  For each pair of files with the same name in both lists, and for
  each file that only exists in one of the lists, it asks the user
  what to do. If the file only exists in one list, the default action
  is to copy it to the other server (or delete it on the other server,
  if the lists says the file was deleted). I.e., the default action
  can be '<' (copy form server to to server 1) or '>' (copy from
  server 1 to server 2) or skip ('/').

  In batch mode, the default action is automatically chosen.

  The user may answer with the following:

  '>'  update on server2
  '<'  update on server1
  '/'  skip, update neither
  'p'  go back to previous pair
  'q'  quit without doing any synchronizations
  'g'  turn on batch_mode for the rest of the files
  'x'  show file dates and sizes
  '?'  show help (handled internally by ask())

  When the user answers 'x', the servers are asked for the file mode,
  modification time and size, which are displayed with
  display_file_info() and then the question what to do with the files
  is repeated.
*/

#include "stdincls.h"
#include "types.e"		/* Defines struct fileinfo */
#include "getline.e"		/* Defines getline_chomp() */
#include "print.e"		/* Defines print() */
#include "c-profile.e"		/* Defines struct Profile */
#include "c-askuser.e"		/* Defines ask() */


/* mode_to_symbolic -- turn mode bits into a string */
static void mode_to_symbolic(const unsigned int mode, char *buf)
{
  /* Assume buf has at least 11 bytes (10 + final '\0') */
  buf[0] = S_ISREG(mode) ? '-' : S_ISLNK(mode) ? 'l' : S_ISSOCK(mode) ? 's' :
    S_ISBLK(mode) ? 'b' : S_ISFIFO(mode) ? 'p' :  S_ISCHR(mode) ? 'c' :
    S_ISDIR(mode) ? 'd' : '?';
  buf[1] = mode & S_IRUSR ? 'r' : '-';
  buf[2] = mode & S_IWUSR ? 'w' : '-';
  buf[3] = mode & S_ISUID && mode & S_IXUSR ? 's' : mode & S_ISUID ? 'S' :
    mode & S_IXUSR ? 'x' : '-';
  buf[4] = mode & S_IRGRP ? 'r' : '-';
  buf[5] = mode & S_IWGRP ? 'w' : '-';
  buf[6] = mode & S_ISGID && mode & S_IXGRP ? 's' : mode & S_ISGID ? 'S' :
    mode & S_IXGRP ? 'x' : '-';
  buf[7] = mode & S_IROTH ? 'r' : '-';
  buf[8] = mode & S_IWOTH ? 'w' : '-';
  buf[9] = mode & S_ISVTX && mode & S_IXOTH ? 't' : mode & S_ISVTX ? 'T' :
    mode & S_IXOTH ? 'x' : '-';
  buf[10] = '\0';
}


/* display_file_info -- display size, date, etc. of a file on both servers */
static void display_file_info(Profile profile, const char *path)
{
  char buf[100], *p, mode[12], *line;
  time_t time;
  long long size;

  /* Send command to servers */
  print(profile->to1, "lstat %s\n", path);
  print(profile->to2, "lstat %s\n", path);

  /* Read and parse the reply */
  if (!(line = getline1(profile->from1)))
    warn(NULL);
  else if (line[0] == '?')
    printf(_("[1] (absent)\n"));
  else {
    mode_to_symbolic(strtoul(line + 2, &p, 8), mode);
    time = strtol(p, &p, 10);
    strftime(buf, sizeof(buf), "%c", localtime(&time));
    size = strtoull(p, &p, 10);
    printf(_("[1] %1$s  %2$-32s  %3$10lld bytes%4$s\n"), mode, buf,
	   (long long)size, mode[0] == 'l' ? _(" (symlink)") : "");
  }

  if (!(line = getline1(profile->from2)))
    warn(NULL);
  else if (line[0] == '?')
    printf(_("[2] (absent)\n"));
  else {
    mode_to_symbolic(strtoul(line + 2, &p, 8), mode);
    time = strtol(p, &p, 10);
    strftime(buf, sizeof(buf), "%c", localtime(&time));
    size = strtoull(p, &p, 10);
    printf(_("[2] %1$s  %2$-32s  %3$10lld bytes%4$s\n"), mode, buf,
	   (long long)size, mode[0] == 'l' ? _(" (symlink)") : "");
  }
}


/* determine_file_actions -- ask user what to do with each file pair */
EXPORT bool determine_file_actions(Profile profile,
				   fileinfo files1[], int nfiles1,
				   fileinfo files2[], int nfiles2,
				   fileinfo *send_to1[], int *nsend1,
				   fileinfo *send_to2[], int *nsend2,
				   char **delete[], int *ndelete)
{
  struct {fileinfo *h1, *h2; char default_action;} *questions;
  int i, j, k, r, with_conflict;
  char default_action;
  fileinfo *h1, *h2;
  int action;

  /* Allocate an array to store the default action for each file. This
     must be on the heap and not on the stack, because it may be
     bigger than the stack. */
  questions = malloc((nfiles1 + nfiles2) * sizeof(*questions));
  if (!questions) {warn(NULL); return false;}

  /* Sort the two lists. They may have been sorted on the servers
     using different collation orders. */
  qsort(files1, nfiles1, sizeof(*files1),
	profile->nocase ? cmpcase_fileinfo : cmp_fileinfo);
  qsort(files2, nfiles2, sizeof(*files2),
	profile->nocase ? cmpcase_fileinfo : cmp_fileinfo);

  /* Determine default actions for each pair */
  for (i = j = k = 0; i < nfiles1 || j < nfiles2; ) {

    if (i == nfiles1) r = 1;
    else if (j == nfiles2) r = -1;
    else if (profile->nocase) r = strcasecmp(files1[i].path, files2[j].path);
    else r = strcmp(files1[i].path, files2[j].path);

    if (!is_excluded(profile, r < 0 ? files1[i].path : files2[j].path)) {
      if (r < 0 && files1[i].status == 'd') {
	/* Deleted on 1, absent on 2, so skip */
      } else if (r < 0) {	/* No such file on root2 */
	questions[k].h1 = files1 + i;
	questions[k].h2 = NULL;
	questions[k++].default_action = '>';
      } else if (r > 0 && files2[j].status == 'd') {
	/* Deleted on 2, absent on 1, so skip */
      } else if (r > 0) {	/* No such file on root1 */
	questions[k].h1 = NULL;
	questions[k].h2 = files2 + j;
	questions[k++].default_action = '<';
      } else if (files1[i].status == 'd' && files2[j].status == 'd') {
	/* Deleted on both sides, skip */
      } else if (files1[i].status == '=' && files2[j].status == '=') {
	/* Unchanged on both sides, skip */
      } else if (files1[i].status == '=') { /* Only root2 changed */
	questions[k].h1 = files1 + i;
	questions[k].h2 = files2 + j;
	questions[k++].default_action = '<';
      } else if (files2[j].status == '=') { /* Only root1 changed */
	questions[k].h1 = files1 + i;
	questions[k].h2 = files2 + j;
	questions[k++].default_action = '>';
      } else { 			/* Both sides changed in some way */
	questions[k].h1 = files1 + i;
	questions[k].h2 = files2 + j;
	questions[k++].default_action = profile->conflict_resolution;
      }
    }

    if (r >= 0) j++;
    if (r <= 0) i++;
  }

  if (!profile->quiet)
    printf(ngettext("Found %d file to synchronize\n\n",
		    "Found %d files to synchronize\n\n", k), k);

  /* Ask the user what to do with each file, unless we're in batch_mode */
  if (!profile->batch_mode) {
    printf("1 = %s\n", profile->root1);
    printf("2 = %s\n", profile->root2);
  }

  /* Run the loop twice, first looking only for files changed on both
     sides, then looking for files only changed on one side. */
  for (with_conflict = 1; !signaled && with_conflict >= 0; with_conflict--) {
    for (i = 0; !signaled && i < k;) {

      h1 = questions[i].h1;
      h2 = questions[i].h2;
      default_action = questions[i].default_action;

      assert(h1 || h2);
      assert(default_action=='>' || default_action=='<' || default_action=='/');

      /* Check if we're presenting conflicts or non-conflicts */
      if ((with_conflict && default_action != '/') ||
	  (!with_conflict && default_action == '/')) {

	i++;			/* Skip this question in this round */

      } else {

	/* Something to do. Ask or infer the action. */
	if (profile->batch_mode) action = default_action;
	else action = ask(h1, h2, default_action);

	while (action == 'x') {	/* User asked for file dates and sizes */
	  display_file_info(profile, h1 ? h1->path : h2->path);
	  action = ask(h1, h2, default_action);
	}

	if (action == 'q') {free(questions); return false;}

	if (action == '>') {
	  if (h1 && h1->status != 'd') {
	    *send_to2 = realloc(*send_to2, (*nsend2 + 1) * sizeof(**send_to2));
	    if (!*send_to2) {warn(NULL); free(questions); return false;}
	    (*send_to2)[(*nsend2)++] = *h1;
	  } else if (!h1 || (h1->status == 'd' && h2)) {
	    *delete = realloc(*delete, (*ndelete + 1) * sizeof(**delete));
	    if (!*delete) {warn(NULL); free(questions); return false;}
	    (*delete)[(*ndelete)++] = h2->path;
	  }
	} else if (action == '<') {
	  if (h2 && h2->status != 'd') {
	    *send_to1 = realloc(*send_to1, (*nsend1 + 1) * sizeof(**send_to1));
	    if (!*send_to1) {warn(NULL); free(questions); return false;}
	    (*send_to1)[(*nsend1)++] = *h2;
	  } else if (!h2 || (h2->status == 'd' && h1)) {
	    *delete = realloc(*delete, (*ndelete + 1) * sizeof(**delete));
	    if (!*delete) {warn(NULL); free(questions); return false;}
	    (*delete)[(*ndelete)++] = h1->path;
	  }
	}

	if (action == 'g') {
	  profile->batch_mode = true; /* And do not increment i */

	} else if (action == 'p') {   /* Back to previous */

	  while (1) {
	    if (i > 0) i--;
	    else if (with_conflict) break; /* Cannot go further back */
	    else {with_conflict = 1; i = k - 1;} /* Back to first batch */
	    if (with_conflict && questions[i].default_action == '/') break;
	    if (!with_conflict && questions[i].default_action != '/') break;
	  }

	} else			/* Move to next files */
	  i++;
      }
    }
  }

  /* If there are files to update and we're not in batch_mode (and not
     interrupted), ask the user for a final confirmation */
  if (*nsend1 == 0 && *nsend2 == 0 && *ndelete == 0) {
    printf(_("Nothing to do.\n"));
  } else if (!profile->batch_mode && !signaled) {
    do {
      printf(_("Synchronize the files? (y/n) "));
      action = get_keypress();
      if (signaled) printf(_("(interrupted)\n")); else printf("%c\n", action);
    } while (!signaled && !strchr("ynYN", action));

    /* If answer is "n", delete the lists */
    if (action == 'n' || action == 'N')
      *nsend1 = *nsend2 = *ndelete = 0;
      /* Don't free(), client() will do that. */
  }

  free(questions);
  return !signaled;		/* True unless a signal handler set a flag. */
}
