/*
  serve() is called by main() when the program is called with option
  "-d" (daemon) and by client() in "c-client.c" to start a server on
  the local host. Its task is to read commands and call the
  corresponding functions. See protocol.html for a description of the
  various commands.
*/

#include "stdincls.h"
#include "types.e"
#include "connectsock.e"
#include "getline.e"
#include "print.e"
#include "errcodes.e"
#include "s-store.e"
#include "s-symlink.e"
#include "s-readlink.e"
#include "s-sayhello.e"
#include "s-list.e"
#include "s-update.e"
#include "s-delta.e"
#include "s-updatelog.e"
#include "s-delete.e"
#include "s-chmod.e"
#include "s-lstat.e"

#define QLEN 3			/* Queue for accepting network connections */


/* serve1 -- loop reading commands until a "quit" or eof */
static void serve1(FILE *in, FILE *out)
{
  bool store_created, stopping = false, case_sensitive, use_shortcut = true;
  char root[PATH_MAX+1] = "", *remote = NULL, *line, *p;
  struct stat statbuf;
  size_t blocksize;
  DB store = NULL;
  fileinfo info;

  /* Read and write one line at a time */
  (void) setlinebuf(in);
  (void) setlinebuf(out);

  /* Announce our version and capabilities */
  say_hello(out, &case_sensitive);

  info.path = NULL;		/* Avoid free() of uninitialized pointer */

  /* Loop reading commands */
  while (!stopping && !signaled && (line = getline_chomp(in))) {

    if (strncmp(line, "remote ", 7) == 0) {
      /* Syntax: remote STRING */
      /* The name of the root we're being sync'ed with. */
      free(remote);
      if (!(remote = strdup(line + 7)) ||
	  (store && !store_close(store)) ||
	  (*root && !(store = store_open(remote, root, &store_created))))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else
	print1(out, "OK\n");

    } else if (strncmp(line, "local ", 6) == 0) {
      /* Syntax: local PATH */
      /* The name of the local file or directory to sync. */
      if (!realpath(line + 6, root) ||
	  stat(root, &statbuf) == -1 ||
	  (store && !store_close(store)) ||
	  (remote && !(store = store_open(remote, root, &store_created))))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else if (S_ISREG(statbuf.st_mode)) print(out, "file %s\n", root);
      else if (!S_ISDIR(statbuf.st_mode)) print(out, "other %s\n", root);
      else {
	if (root[2]) strcat(root, "/"); /* Directory always ends in '/' */
	print(out, "directory %s\n", root);
      }

    } else if (strcmp(line, "list") == 0) {
      /* Send a list of new, updated and deleted files */
      if (!remote) print(out, "? %03d Missing `remote' command\n", EC_REMOTE);
      else if (!*root) print(out, "? %03d Missing `local' command\n", EC_LOCAL);
      else if (!store) print(out, "? %03d No logs available\n", EC_NOLOG);
      else {
	print1(out, store_created ? "creating\n" : "comparing\n");
	debug("calling list()\n");
	list(out, store, root, case_sensitive);
      }

    } else if (strncmp(line, "lstat ", 6) == 0) {
      /* Syntax: lstat PATH */
      /* Get mode, time and size of the given file */
      if (!(info.path = strdup(line + 6)))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else if (!*root)
	print(out, "? %03d missing `lcoal' command\n", EC_LOCAL);
      else
	get_lstat(out, root, info.path);
      free(info.path);

    } else if (strncmp(line, "update ", 7) == 0) {
      /* Update a file using the rsync algorithm */
      info.status = '+';	/* Indicate we will use the info.sums field */
      if (!remote)
	print(out,"? %03d Missing `remote' command\n", EC_REMOTE);
      else if (!*root)
	print(out, "? %03d Missing `local' command\n", EC_LOCAL);
      else if ((blocksize = strtoul(line + 7, &p, 10)) == 0)
	print(out, "? %03d Missing or incorrect blocksize\n", EC_SYNTAX);
      else if ((info.mode = strtoul(p, &p, 8)) > 07777)
	print(out, "? %03d Incorrect mode\n", EC_SYNTAX);
      else if ((info.time = strtol(p, &p, 10)), *p != ' ')
	print(out, "? %03d Missing size\n", EC_SYNTAX);
      else if ((info.size = strtoul(p, &p, 10)), *p != ' ')
	print(out, "? %03d Missing checksum\n", EC_SYNTAX);
      else if ((info.sums.sum = strtoul(p, &p, 16)), *p != ' ')
	print(out, "? %03d Missing digest\n", EC_SYNTAX);
      else if (!str_to_digest(p + 1, info.sums.digest, &p))
	print(out, "? %03d Incorrect digest\n", EC_SYNTAX);
      else if (*p != ' ')
	print(out, "? %03d Missing path\n", EC_SYNTAX); 
      else if (!(info.path = strdup(p + 1)))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else
	update(in, out, root, store, blocksize, info, case_sensitive);
      free(info.path);

    } else if (strncmp(line, "update0 ", 8) == 0) {
      /* Update a file using the rsync algorithm, no file checksum/digest */
      info.status = '-';	/* Indicate we won't use the info.sums field */
      if (!remote)
	print(out,"? %03d missing `remote' command\n", EC_REMOTE);
      else if (!*root)
	print(out, "? %03d missing `local' command\n", EC_LOCAL);
      else if ((blocksize = strtoul(line + 8, &p, 10)) == 0)
	print(out, "? %03d missing or incorrect blocksize\n", EC_SYNTAX);
      else if ((info.mode = strtoul(p, &p, 8)) > 07777)
	print(out, "? %03d incorrect mode\n", EC_SYNTAX);
      else if ((info.time = strtol(p, &p, 10)), *p != ' ')
	print(out, "? %03d missing size\n", EC_SYNTAX);
      else if ((info.size = strtoul(p, &p, 10)), *p != ' ')
	print(out, "? %03d missing path\n", EC_SYNTAX);
      else if (!(info.path = strdup(p + 1)))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else
	update(in, out, root, store, blocksize, info, case_sensitive);
      free(info.path);

    } else if (strncmp(line, "log ", 4) == 0) {
      /* Update an entry in the log */
      info.status = '-';	/* Indicate we won't use the info.sums field */
      if (!remote)
	print(out,"? %03d missing `remote' command\n", EC_REMOTE);
      else if (!*root)
	print(out, "? %03d missing `local' command\n", EC_LOCAL);
      else if ((info.mode = strtoul(line + 4, &p, 8)) > 0177777)
	print(out, "? %03d incorrect mode\n", EC_SYNTAX);
      else if ((info.time = strtol(p, &p, 10)), *p != ' ')
	print(out, "? %03d missing size\n", EC_SYNTAX);
      else if ((info.size = strtoul(p, &p, 10)), *p != ' ')
	print(out, "? %03d missing path\n", EC_SYNTAX);
      else if (!(info.path = strdup(p + 1)))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else
	update_log(out, root, store, info);
      free(info.path);

    } else if (strncmp(line, "del ", 4) == 0) {
      /* Delete the indicated file */
      if (!remote)
	print(out, "? %03d missing `remote' command\n", EC_REMOTE);
      else if (!*root)
	print(out, "? %03d missing `local' command\n", EC_LOCAL);
      else if (!(info.path = strdup(line + 4)))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else
	delete_file(out, root, info.path, store);
      free(info.path);

    } else if (strncmp(line, "delta ", 6) == 0) {
      /* Compute the delta given a file and the signature of another */
      if (!*root)
	print(out, "? %03d missing `local' command\n", EC_LOCAL);
      else if ((blocksize = strtoul(line + 6, &p, 10)) == 0)
	print(out, "? %03d Incorrect blocksize\n", EC_SYNTAX);
      else if (*p != ' ')
	print(out, "? %03d Missing path\n", EC_SYNTAX);
      else if (!(info.path = strdup(p + 1)))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else
	compute_delta(in, out, root, blocksize, info.path, use_shortcut);
      free(info.path);

    } else if (strncmp(line, "chmod ", 6) == 0) {
      /* Change the permission bits of a file */
      if (!*root)
	print(out, "? %03d missing `local' command\n", EC_LOCAL);
      else if ((info.mode = strtoul(line + 6, &p, 8)) > 07777)
	print(out, "? %03d illegal mode\n", EC_MODE);
      else if (!(info.path = strdup(p + 1)))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else
	change_mode(out, root, info.mode, info.path, store);
      free(info.path);

    } else if (strncmp(line, "symlink ", 8) == 0) {
      /* Make path into a symlink */
      if (!*root)
	print(out, "? %03d missing `local' command\n", EC_LOCAL);
      else if ((info.time = strtoul(line + 8, &p, 10)), *p != ' ')
	print(out, "? %03d Incorrect time or missing path\n", EC_SYNTAX);
      else if (!(info.path = strdup(p + 1)))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else
	make_symlink(in, out, root, info.time, info.path, store);
      free(info.path);

    } else if (strncmp(line, "readlink ", 9) == 0) {
      /* Syntax: readlink PATH */
      /* Return what symbolic link PATH points to */
      if (!*root)
	print(out, "? %03d missing `local' command\n", EC_LOCAL);
      else if (!(info.path = strdup(line + 9)))
	print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      else
	read_link(out, root, info.path);
      free(info.path);

    } else if (strncmp(line, "version ", 8) == 0) {
      /* Syntax: version N [noshortcuts]*/
      /* Set the protocol to use */
      int v = strtol(line + 8, &p, 10);
      if (v == 0 || v > R2SYNCPROTO)
	print(out, "? %03d Unknown version\n", EC_VERSION);
      else {
	use_shortcut = strcmp(p, " noshortcuts") != 0;
	print1(out, "OK\n");
      }

    } else if (strcmp(line, "reset") == 0) {
      /* Syntax: reset */
      /* Reset the log, making all files appear as new */
      if (!remote) print(out, "? %03d missing `remote' command\n", EC_REMOTE);
      else if (!*root) print(out, "? %03d missing `local' command\n",EC_LOCAL);
      else {store_reset(store); print1(out, "OK\n");}

    } else if (strcmp(line, "quit") == 0) {
      /* Syntax: quit */
      /* Stop the server */
      stopping = true;

    } else {
      print(out, "? %03d Unknown command or syntax error\n", EC_COMMAND);
    }
  }

  if (!stopping && !feof(in) && !signaled) warn(NULL);

  free(remote);

  if (store && !store_close(store)) warn(NULL);
}


/* serve -- read and and execute commands from input */
EXPORT bool serve(const int input, const int output, const char * const port)
{
  int fd, socket;
  FILE *in, *out;

  debug("Starting serve()\n");

  /* If a port or service was passed in, create a listening socket and
     repeatedly accept connections. Otherwise, run the command loop
     only once, with the given file descriptors for input and output.
  */
  if (port) {
    if ((socket = passiveTCP(port, QLEN)) == -1) {
      warn(NULL);
      return false;
    }
    while (!signaled)
      if ((fd = accept(socket, NULL, NULL)) == -1 || !(in = fdopen(fd, "r+")))
	warn(NULL);
      else {
	serve1(in, in);
	(void) fclose(in);
      }
  } else {
    if (!(in = fdopen(input, "r"))) return false;
    if (!(out = fdopen(output, "w"))) return false;
    serve1(in, out);
    (void) fclose(out);
    (void) fclose(in);
  }

  debug("Ending serve()\n");

  return !signaled;
}
