/*
  checksum() is called by compute_signature() and by compute_delta()
  to compute the fast but weak checksum of a block of data.

  checksum_remove_one() and checksum_add_one() are called by
  compute_delta() to compute the checksum of the block data[i+1..j+1]
  from the known checksum for data[i..j]

  Implementation in protocol version 1:

  The checksum is the same one as used in rsync 3.1.2 (rsync protocol
  version 31), which is based on Adler-32, but with computations modulo
  2^16 instead of modulo 65521 (a prime number). That avoids having to
  actually compute the modulo. Instead, bits other than the 16 least
  significant are simply discarded.

  That is, the checksum of the sequence of N bytes d[0]...d[N-1] is
  defined as C = 2^16 * B + A, where A = sum(i=0, i=N-1, d[i]) mod
  2^16 and B = sum(i=0, i=N-1, (N - i) * d[i]) mod 2^16.
 */

#include "stdincls.h"

struct checksum_state {uint32_t s;};
EXPORT typedef struct checksum_state *checksum_state;


/* checksum_init -- initialize a state for a progressive checksum computation */
EXPORT void checksum_init(checksum_state *state)
{
  *state = malloc(sizeof(**state));
  (*state)->s = 0;
}


/* checksum_update -- add another block to the checksum */
EXPORT void checksum_update(checksum_state state, const unsigned char *data,
			    size_t len)
{
  unsigned long j;
  uint32_t a, b;

  a = state->s & 0xFFFF;		/* Split the sum back into a and b */
  b = state->s >> 16;
  for (j = 0; j < len; j++) {a += data[j]; b += a;}
  state->s = (b << 16) | (a & 0xFFFF);
}


/* checksum_final -- extract the checksum from a state and destroy the state */
EXPORT void checksum_final(checksum_state state, uint32_t *sum)
{
  *sum = state->s;
  free(state);
}


/* checksum -- compute a weak checksum of a circular block of bytes */
EXPORT uint32_t checksum(unsigned const char *data, size_t datasize,
			 unsigned long start, unsigned long len)
{
  /* "Circular block" means that data wraps around at the end. I.e.,
   * the bytes are
   *
   *   data[start]...data[datasize-1], data[0]...data[start-1].
   *
   * Only the first len <= datasize bytes of the block are used, i.e.,
   * data[start]...data[(start+len)%datasize]]
   */
  uint32_t a, b;
  unsigned long j;

  if (start + len > datasize) {
    for (a = b = 0, j = start; j < datasize; j++) {a += data[j]; b += a;}
    for (j = 0; j < start + len - datasize; j++) {a += data[j]; b += a;}
  } else
    for (a = b = 0, j = start; j < start + len; j++) {a += data[j]; b += a;}
  return (b << 16) | (a & 0xFFFF);
}


/* checksum_remove_one -- remove 1st byte of block from the checksum */
EXPORT uint32_t checksum_remove_one(uint32_t oldsum, unsigned char remove,
				    unsigned long len)
{
  uint32_t a, b;

  a = oldsum & 0xFFFF;		/* Split the sum back into a and b */
  b = oldsum >> 16;
  a -= remove;			/* Update a and b */
  b -= len * remove;
  return (b << 16) | (a & 0xFFFF); /* Recombine a and b into a new checksum */
}


/* checksum_add_one -- add a new last byte to the checksum */
EXPORT uint32_t checksum_add_one(uint32_t oldsum, unsigned char add)
{
  uint32_t a, b;

  a = oldsum & 0xFFFF;		/* Split the sum back into a and b */
  b = oldsum >> 16;
  a += add;			/* Update a and b */
  b += a;
  return (b << 16) | (a & 0xFFFF); /* Recombine a and b into a new checksum */
}
